const skillButtons = Array.from(document.getElementsByClassName('skill-button'));
const abilityButtons = Array.from(document.getElementsByClassName('ability-button'));
const weaponButtons = Array.from(document.getElementsByClassName('weapon-button'));
const rollButtons = Array.from(document.getElementsByClassName('roll-button'));
const rawButtons = Array.from(document.getElementsByClassName('raw-button'));

const onPredicate = (predicate, callback, attempts = 10, timeout = 100) => {
    if (predicate()) {
        callback();
    } else if (attempts <= 0) {
        console.warn("Maximum number of attempts exceeded");
    } else {
        setTimeout(function() {
            onPredicate(predicate, callback, attempts - 1, timeout * 2);
        }, timeout);
    }
};

const onElementLoad = (selector, callback) => onPredicate(() => !!document.querySelector(selector), callback);

if (skillButtons.length) {
    skillButtons.forEach(element => {
        element.addEventListener('click', () => {
            const skillValue = document.getElementsByName(element.dataset.value).item(0).value;
            const skillName = element.dataset.label

            chrome.runtime.sendMessage({
                type: 'skill',
                rollValue: skillValue || -1,
                skillName: skillName,
            }, function(response) {
                console.log('sent skill roll');
            });
        });
    });
};

if (abilityButtons.length) {
    abilityButtons.forEach(element => {
        element.addEventListener('click', () => {
            const abilityName = document.getElementsByName(`${element.dataset.value}-name`).item(0).value;
            const abilityDescription = document.getElementsByName(element.dataset.value)?.item(0)?.value;
            const prefix = element.dataset.prefix;

            chrome.runtime.sendMessage({
                type: 'ability',
                prefix,
                abilityName,
                abilityDescription,
            }, function() {
                console.log('sent ability roll');
            });
        });
    });
};

if (weaponButtons.length) {
    weaponButtons.forEach(element => {
        element.addEventListener('click', () => {
            const weaponName = document.getElementsByName(`${element.dataset.value}-name`).item(0).value;
            const weaponBonus = document.getElementsByName(`${element.dataset.value}-bonus`)?.item(0)?.value || 0;
            const skillBonus = document.getElementsByName('ability-weapon-use')?.item(0)?.value || 0;

            chrome.runtime.sendMessage({
                type: 'weapon',
                weaponName,
                weaponBonus,
                skillBonus
            }, function() {
                console.log('sent weapon roll');
            });
        });
    })
}

if (rollButtons.length) {
    rollButtons.forEach(element => {
        element.addEventListener('click', () => {
            const rollName = element.dataset.prefix;
            const rollValue = document.getElementsByName(`${element.dataset.value}`)?.item(0)?.value || element.dataset.roll;

            chrome.runtime.sendMessage({
                type: 'roll',
                rollName,
                rollValue
            }, function() {
                console.log('sent custom roll');
            });
        });
    })
}

if (rawButtons.length) {
    rawButtons.forEach(element => {
        element.addEventListener('click', () => {
            const text = element.dataset.text;

            chrome.runtime.sendMessage({
                type: 'raw',
                text,
            }, function() {
                console.log('sent raw text');
            });
        });
    })
}

const sendCustomRollMessage = (rollName, rollValue) => {
    const input = document.querySelector("#textchat-input");

    if (input){
        input.querySelector("textarea").value = `&{template:default} {{name=${rollName}}} {{Dobás Eredménye:=[[${rollValue}]]}}`
        input.querySelector(".btn").click();
        return;
    }

    const chatMessage = document.querySelector("#chat-message");
    
    if (chatMessage) {
        chatMessage.value = `/roll ${rollValue} #<h2>${rollName}</h2>`;
        // Focus the textarea first, then trigger Enter key
        chatMessage.focus();
        setTimeout(() => {
            chatMessage.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
            chatMessage.dispatchEvent(new KeyboardEvent('keyup', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
        }, 10);
    }

    console.debug("rolled in VTT");
};

const sendWeaponHitMessage = (weaponName, weaponBonus, skillBonus) => {
    var modifier = '';

    if (weaponBonus) modifier+= `+${weaponBonus}`;
    if (skillBonus) modifier+= `+${skillBonus}`;

    const input = document.querySelector("#textchat-input");

    if (input){
        input.querySelector("textarea").value = `&{template:default} {{name=${weaponName}}} {{Dobás Eredménye:=[[3d6${modifier}]]}}`
        input.querySelector(".btn").click();
        return;
    }

    const chatMessage = document.querySelector("#chat-message");
    
    if (chatMessage) {
        chatMessage.value = `/roll 3d6${modifier} #<h3>${weaponName}</h3>`;
        chatMessage.focus();
        setTimeout(() => {
            chatMessage.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
            chatMessage.dispatchEvent(new KeyboardEvent('keyup', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
        }, 10);
    }

    console.debug("rolled in VTT");
};

const sendSkillMessage = async (skillName, value) => {
    const modifier = value && value !== -1 ? `+${value}` : '';
    const input = document.querySelector("#textchat-input");

    if (input) {
        input.querySelector("textarea").value = `&{template:default} {{name=${skillName}}} {{Dobás Eredménye:=[[3d6${modifier}]]}}`
        input.querySelector(".btn").click();
        return;
    }

    const chatMessage = document.querySelector("#chat-message");
    
    if (chatMessage) {
        chatMessage.value = `/roll 3d6${modifier} #<h2>${skillName}</h2>`;
        // Focus the textarea first, then trigger Enter key
        chatMessage.focus();
        setTimeout(() => {
            chatMessage.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
            chatMessage.dispatchEvent(new KeyboardEvent('keyup', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
        }, 10);
    }

    console.debug("rolled in VTT");
};

const sendAbilityMessage = (prefix, name, description) => {
    const input = document.querySelector("#textchat-input");
  
    if (input) {
        input.querySelector("textarea").value = `&{template:default} {{name=${prefix}: ${name}}}\n${description}`
        input.querySelector(".btn").click();
        return;
    }

    const chatMessage = document.querySelector("#chat-message");
    
    if (chatMessage) {
        chatMessage.value = `<h3>${prefix}: ${name}</h3><p>${description}</p>`;
        chatMessage.focus();
        setTimeout(() => {
            chatMessage.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
            chatMessage.dispatchEvent(new KeyboardEvent('keyup', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
        }, 10);
    }

    console.debug("rolled in VTT");
};

const sendRawMessage = (text) => {
    const input = document.querySelector("#textchat-input");

    if (input){
        input.querySelector("textarea").value = text;
        input.querySelector(".btn").click();
        return;
    }

    const chatMessage = document.querySelector("#chat-message");
    
    if (chatMessage) {
        chatMessage.value = `<p>${text}</p>`;
        chatMessage.focus();
        setTimeout(() => {
            chatMessage.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
            chatMessage.dispatchEvent(new KeyboardEvent('keyup', { key: 'Enter', code: 'Enter', keyCode: 13, which: 13, bubbles: true, cancelable: true }));
        }, 10);
    }

    console.debug("rolled in VTT");
};


chrome.runtime.onMessage.addListener(
    function(request) {
        if (request.type === 'skill') {
            sendSkillMessage(request.skillName, request.rollValue);
        }

        if (request.type === 'ability') {
            sendAbilityMessage(request.prefix, request.abilityName, request.abilityDescription || '');
        }
        
        if (request.type === 'weapon') {
            sendWeaponHitMessage(request.weaponName, request.weaponBonus, request.skillBonus);
        }

        if (request.type === 'roll') {
            sendCustomRollMessage(request.rollName, request.rollValue);
        }

        if (request.type === 'raw') {
            sendRawMessage(request.text);
        }
    }
);